﻿namespace Microsoft.Samples.PlanMyNight.Data
{
    using System;
    using System.Collections.Generic;
    using System.Data.Objects;
    using System.Linq;
    using Microsoft.Samples.PlanMyNight.Entities;

    public class ItinerariesRepository : IItinerariesRepository
    {
        public PagingResult<Itinerary> SearchByActivity(string activityId, int pageSize, int pageNumber)
        {
            using (var ctx = new PlanMyNightEntities())
            {
                ctx.ContextOptions.ProxyCreationEnabled = false;

                var query = from itinerary in ctx.Itineraries.Include("Activities")
                            where itinerary.Activities.Any(t => t.ActivityId == activityId)
                                  && itinerary.IsPublic
                            orderby itinerary.Rating
                            select itinerary;

                return PageResults(query, pageNumber, pageSize);
            }

        }

        public PagingResult<Itinerary> SearchByZipCode(int activityTypeId, string zip, int pageSize, int pageNumber)
        {
            using (var ctx = new PlanMyNightEntities())
            {
                ctx.ContextOptions.ProxyCreationEnabled = false;

                var query = from itinerary in ctx.Itineraries.Include("Activities")
                            where itinerary.Activities.Any(t => t.TypeId == activityTypeId && t.Zip == zip)
                                  && itinerary.IsPublic
                            orderby itinerary.Rating
                            select itinerary;

                return PageResults(query, pageNumber, pageSize);
            }

        }

        public PagingResult<Itinerary> SearchByCity(int activityTypeId, string state, string city, int pageSize, int pageNumber)
        {
            using (var ctx = new PlanMyNightEntities())
            {
                ctx.ContextOptions.ProxyCreationEnabled = false;

                var query = from itinerary in ctx.Itineraries.Include("Activities")
                            where itinerary.Activities.Any(t => t.TypeId == activityTypeId && t.State == state && t.City.Equals(city, StringComparison.OrdinalIgnoreCase))
                             && itinerary.IsPublic
                            orderby itinerary.Rating
                            select itinerary;

                return PageResults(query, pageNumber, pageSize);
            }
        }

        public PagingResult<Itinerary> SearchByRadius(int activityTypeId, double longitude, double latitude, double radius, int pageSize, int pageNumber)
        {
            using (var ctx = new PlanMyNightEntities())
            {
                ctx.ContextOptions.ProxyCreationEnabled = false;

                // Stored Procedure with output parameter
                var totalOutput = new ObjectParameter("total", typeof(int));
                var items = ctx.RetrieveItinerariesWithinArea(activityTypeId, latitude, longitude, radius, pageSize, pageNumber, totalOutput).ToArray();

                foreach (var item in items)
                {
                    item.Activities.ToList().AddRange(this.Retrieve(item.Id).Activities);
                }

                int total = totalOutput.Value == DBNull.Value ? 0 : (int)totalOutput.Value;

                return new PagingResult<Itinerary>(items)
                {
                    TotalItems = total,
                    PageSize = pageSize,
                    CurrentPage = pageNumber
                };
            }

        }

        public void Add(Itinerary itinerary)
        {
            using (var ctx = new PlanMyNightEntities())
            {
                ctx.Itineraries.AddObject(itinerary);
                ctx.SaveChanges();
            }

        }

        public Itinerary Retrieve(long itineraryId)
        {
            using (var ctx = new PlanMyNightEntities())
            {
                ctx.ContextOptions.ProxyCreationEnabled = false;
                var itinerary = ctx.Itineraries
                          .Include("Activities")
                          .Where(i => i.Id == itineraryId)
                          .SingleOrDefault();

                if (itinerary != null)
                {
                    var activities = itinerary.Activities.OrderBy(a => a.Order).ToArray();
                    itinerary.Activities.Clear();

                    foreach (var activity in activities)
                    {
                        itinerary.Activities.Add(activity);
                    }
                }

                return itinerary;
            }
        }

        public IEnumerable<Itinerary> RetrieveByUser(Guid userId)
        {
            using (var ctx = new PlanMyNightEntities())
            {
                ctx.ContextOptions.ProxyCreationEnabled = false;
                var userItineraries = ctx.Itineraries
                                         .Include("Activities") 
                                         .Where(i => i.UserId == userId)
                                         .OrderBy(i => i.Name);

                foreach (var item in userItineraries)
                {
                    var activities = item.Activities.OrderBy(a => a.Order).ToArray();
                    item.Activities.Clear();
                    foreach (var activity in activities)
                    {
                        item.Activities.Add(activity);
                    }
                }

                return userItineraries.ToList();
            }
        }

        public void Update(Itinerary itinerary)
        {
            using (var ctx = new PlanMyNightEntities())
            {
                var original = ctx.Itineraries.Single(i => i.Id == itinerary.Id);
                original.Activities.Clear();
                foreach (var item in itinerary.Activities)
                {
                    original.Activities.Add(new ItineraryActivity
                    {
                        ItineraryId = item.ItineraryId,
                        ActivityId = item.ActivityId,
                        Order = item.Order,
                        EstimatedMinutes = item.EstimatedMinutes,
                        State = item.State,
                        City = item.City,
                        TypeId = item.TypeId,
                        Zip = item.Zip,
                        Longitude = item.Longitude,
                        Latitude = item.Latitude,
                    });
                }

                original.Name = itinerary.Name;
                original.Description = itinerary.Description;
                original.IsPublic = itinerary.IsPublic;

                ctx.SaveChanges();
            }
        }

        public string GetUserDisplayName(Guid userId)
        {
            using (var ctx = new PlanMyNightEntities())
            {
                var user = ctx.UserProfiles.FirstOrDefault(u => u.UserName == userId);
                if (user != null)
                {
                    return user.FullName;
                }
                else
                {
                    return null;
                }
            }
        }

        public bool CanUserRateItinerary(long itineraryId, Guid userId)
        {
            using (var ctx = new PlanMyNightEntities())
            {
                return !ctx.ItineraryRatings.Any(r => r.ItineraryId == itineraryId && r.UserId == userId);
            }
        }

        public void RateItinerary(long itineraryId, Guid userId, byte rating, DateTime timestamp)
        {
            using (var ctx = new PlanMyNightEntities())
            {
                var ratingRow = new ItineraryRating
                {
                    ItineraryId = itineraryId,
                    UserId = userId,
                    Rating = rating,
                    Timestamp = timestamp
                };

                ctx.ItineraryRatings.AddObject(ratingRow);
                ctx.SaveChanges();
            }
        }

        public IEnumerable<ItineraryComment> RetrieveComments(long itineraryId)
        {
            using (var ctx = new PlanMyNightEntities())
            {
                var reviews = from c in ctx.ItineraryComments
                              join p in ctx.UserProfiles on c.UserId equals p.UserName
                              where c.ItineraryId == itineraryId
                              orderby c.Timestamp
                              select new { Id = c.Id, Body = c.Body, Timestamp = c.Timestamp, UserId = c.UserId, IpAddress = c.IpAddress, ItineraryId = c.ItineraryId, DisplayName = p.FullName };
                return reviews
                    .ToList()
                    .Select(c => new ItineraryComment { Id = c.Id, Body = c.Body, Timestamp = c.Timestamp, UserId = c.UserId, IpAddress = c.IpAddress, ItineraryId = c.ItineraryId, DisplayName = c.DisplayName })
                    .ToList();
            }
        }

        public void AddComment(ItineraryComment comment)
        {
            using (var ctx = new PlanMyNightEntities())
            {
                ctx.ItineraryComments.AddObject(comment);
                ctx.SaveChanges();
            }
        }

        private static PagingResult<Itinerary> PageResults(IQueryable<Itinerary> query, int page, int pageSize)
        {
            int rowCount = rowCount = query.Count();
            if (pageSize > 0)
            {
                query = query.Skip((page - 1) * pageSize)
                             .Take(pageSize);
            }

            var result = new PagingResult<Itinerary>(query.ToArray())
            {
                PageSize = pageSize,
                CurrentPage = page,
                TotalItems = rowCount
            };

            return result;
        }
    }
}
